<?php
// Módulo: Reportes
$page_title = 'Reportes y Métricas';

// Ruta corregida para incluir el header.
// Sube dos niveles desde /modulos/reportes/ para llegar a public_html y luego entra a templates.
require_once dirname(dirname(__DIR__)) . '/templates/header.php';

// --- LÓGICA DE DATOS (CONSULTAS A LA BD) ---
// El $conn y $empresa_id ya están disponibles desde el header.

// KPI: Casos por Estado
$stmt_estados = $conn->prepare("SELECT estado, COUNT(id) as total FROM casos WHERE empresa_id = ? GROUP BY estado");
$stmt_estados->bind_param("i", $empresa_id);
$stmt_estados->execute();
$casos_por_estado_raw = $stmt_estados->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt_estados->close();

$casos_por_estado = ['nuevo' => 0, 'respondido' => 0, 'cerrado' => 0, 'en proceso' => 0];
$labels_estados = [];
$datos_estados = [];
foreach($casos_por_estado_raw as $item) {
    if (array_key_exists($item['estado'], $casos_por_estado)) {
        $casos_por_estado[$item['estado']] = $item['total'];
    }
    $labels_estados[] = ucfirst($item['estado']);
    $datos_estados[] = $item['total'];
}

// KPI: Casos por Tipo
$stmt_tipos = $conn->prepare("SELECT tipo_caso, COUNT(id) as total FROM casos WHERE empresa_id = ? GROUP BY tipo_caso");
$stmt_tipos->bind_param("i", $empresa_id);
$stmt_tipos->execute();
$casos_por_tipo_tabla = $stmt_tipos->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt_tipos->close();

$labels_tipos = [];
$datos_tipos = [];
foreach ($casos_por_tipo_tabla as $tipo) {
    $labels_tipos[] = ucfirst($tipo['tipo_caso']);
    $datos_tipos[] = (int)$tipo['total'];
}

// KPI: Totales generales
$total_casos = array_sum($casos_por_estado);
$stmt_clientes = $conn->prepare("SELECT COUNT(id) as total FROM clientes WHERE empresa_id = ?");
$stmt_clientes->bind_param("i", $empresa_id);
$stmt_clientes->execute();
$total_clientes = $stmt_clientes->get_result()->fetch_assoc()['total'];
$stmt_clientes->close();

// KPI: Casos creados en los últimos 30 días
$stmt_ultimos_30 = $conn->prepare("SELECT COUNT(id) as total FROM casos WHERE empresa_id = ? AND fecha_creacion >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
$stmt_ultimos_30->bind_param("i", $empresa_id);
$stmt_ultimos_30->execute();
$casos_ultimos_30_dias = $stmt_ultimos_30->get_result()->fetch_assoc()['total'];
$stmt_ultimos_30->close();

// KPI: Tiempo promedio de primera respuesta
$stmt_tiempo_respuesta = $conn->prepare("
    SELECT AVG(TIMESTAMPDIFF(MINUTE, c.fecha_creacion, r.primera_respuesta)) as tiempo_promedio_minutos
    FROM casos c
    JOIN (
        SELECT caso_id, MIN(fecha_respuesta) as primera_respuesta
        FROM respuestas
        GROUP BY caso_id
    ) r ON c.id = r.caso_id
    WHERE c.empresa_id = ? AND c.estado IN ('respondido', 'cerrado')
");
$stmt_tiempo_respuesta->bind_param("i", $empresa_id);
$stmt_tiempo_respuesta->execute();
$resultado_tiempo = $stmt_tiempo_respuesta->get_result()->fetch_assoc();
$tiempo_promedio_respuesta = $resultado_tiempo['tiempo_promedio_minutos'] ? round($resultado_tiempo['tiempo_promedio_minutos']) : 0;
$stmt_tiempo_respuesta->close();

// --- PREPARACIÓN DE DATOS PARA JAVASCRIPT ---
$json_labels_estados = json_encode($labels_estados);
$json_datos_estados = json_encode($datos_estados);
$json_labels_tipos = json_encode($labels_tipos);
$json_datos_tipos = json_encode($datos_tipos);
?>

<!-- Aquí empieza el contenido HTML específico de la página -->
<div class="header">
    <h1>Reportes y Métricas</h1>
    <p>Estado general de la gestión de reclamos.</p>
</div>

<div class="kpi-grid">
    <div class="card kpi-card"><h2><?= $total_casos ?></h2><p>Casos Totales</p></div>
    <div class="card kpi-card"><h2><?= $casos_por_estado['nuevo'] ?? 0 ?></h2><p>Casos Nuevos</p></div>
    <div class="card kpi-card"><h2><?= $casos_por_estado['respondido'] ?? 0 ?></h2><p>Casos Respondidos</p></div>
    <div class="card kpi-card"><h2><?= $total_clientes ?></h2><p>Clientes Totales</p></div>
    <div class="card kpi-card"><h2><?= $casos_ultimos_30_dias ?></h2><p>Casos (Últ. 30 días)</p></div>
    <div class="card kpi-card">
        <h2>
            <?php
                if ($tiempo_promedio_respuesta >= 60) {
                    echo round($tiempo_promedio_respuesta / 60, 1) . ' <span style="font-size: 0.6em; font-weight: normal;">hrs</span>';
                } else {
                    echo $tiempo_promedio_respuesta . ' <span style="font-size: 0.6em; font-weight: normal;">min</span>';
                }
            ?>
        </h2>
        <p>Tpo. Prom. Respuesta</p>
    </div>
</div>

<div class="charts-grid">
    <div class="chart-container">
        <h3>Casos por Estado</h3>
        <canvas id="graficoCasosEstado"></canvas>
    </div>
    <div class="chart-container">
        <h3>Casos por Tipo</h3>
        <canvas id="graficoCasosTipo"></canvas>
    </div>
</div>

<div class="card" style="margin-top: 2rem;">
    <h3>Desglose de Casos por Tipo</h3>
    <table class="table">
        <thead>
            <tr>
                <th>Tipo de Caso</th>
                <th>Cantidad Total</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($casos_por_tipo_tabla)): ?>
                <tr><td colspan="2" style="text-align: center; padding: 20px;">No hay datos de tipos de casos para mostrar.</td></tr>
            <?php else: ?>
                <?php foreach ($casos_por_tipo_tabla as $tipo): ?>
                <tr>
                    <td><?= ucfirst(sanitizar_html($tipo['tipo_caso'])) ?></td>
                    <td><?= $tipo['total'] ?></td>
                </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<!-- Scripts de Chart.js -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function () {
    const chartDefaults = { responsive: true, plugins: { legend: { position: 'top' } } };

    // GRÁFICO DE BARRAS
    const ctx_estados = document.getElementById('graficoCasosEstado').getContext('2d');
    new Chart(ctx_estados, {
        type: 'bar',
        data: {
            labels: <?= $json_labels_estados ?>,
            datasets: [{
                label: 'Número de Casos',
                data: <?= $json_datos_estados ?>,
                backgroundColor: 'rgba(54, 162, 235, 0.6)',
                borderColor: 'rgba(54, 162, 235, 1)',
                borderWidth: 1
            }]
        },
        options: { ...chartDefaults, scales: { y: { beginAtZero: true, ticks: { stepSize: 1 } } } }
    });

    // GRÁFICO DE TORTA
    const ctx_tipos = document.getElementById('graficoCasosTipo').getContext('2d');
    new Chart(ctx_tipos, {
        type: 'pie',
        data: {
            labels: <?= $json_labels_tipos ?>,
            datasets: [{
                data: <?= $json_datos_tipos ?>,
                backgroundColor: ['#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0', '#9966FF', '#FF9F40'],
                hoverOffset: 4
            }]
        },
        options: { ...chartDefaults }
    });
});
</script>

<?php
// Incluimos el footer que cierra la página y la conexión a la BD
require_once ROOT_PATH . '/templates/footer.php';
?>

